package com.transistorsoft.locationmanager.data.sqlite;

import java.util.ArrayList;
import java.util.List;

import org.json.JSONObject;
import org.json.JSONException;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.text.TextUtils;
import android.util.Log;

import com.transistorsoft.locationmanager.data.LocationDAO;
import com.transistorsoft.locationmanager.data.TSLocation;

public class SQLiteLocationDAO implements LocationDAO {
	private static final String TAG = "TSLocationManager";
	private Context context;
	
	public SQLiteLocationDAO(Context context) {
		this.context = context;
	}
	
	public List<TSLocation> all() {
		SQLiteDatabase db = null;
		Cursor c = null;
		List<TSLocation> rs = new ArrayList<TSLocation>();
		try {
			db = new LocationOpenHelper(context).getReadableDatabase();
			c = db.query(false, LocationOpenHelper.LOCATION_TABLE_NAME, null, null, null, null, null, null, null);
			while (c.moveToNext()) {
				rs.add(hydrate(c));
			}
		} finally {
			if (c != null) {
				c.close();
			}
			if (db != null) {
				db.close();
			}
		}
		return rs;
	}
	/**
	 * Fetches all unlocked records.  Locks those which it selected.
	 */
	public List<TSLocation> allWithLocking() {
		SQLiteDatabase db = null;
		Cursor c = null;
		List<TSLocation> rs = new ArrayList<TSLocation>();
		try {
			db = new LocationOpenHelper(context).getWritableDatabase();
			c = db.query(false, LocationOpenHelper.LOCATION_TABLE_NAME, null, "locked=0", null, null, null, null, null);
			List<Integer> ids = new ArrayList<Integer>();
			while (c.moveToNext()) {
				ids.add(c.getInt(0));
				rs.add(hydrate(c));
			}
			// Now lock all those records we've selected so that another AsyncTask cannot mess with them (ie: double POST
			ContentValues values = new ContentValues();
			values.put("locked", 1);
			int result = db.update(LocationOpenHelper.LOCATION_TABLE_NAME, values, "id IN (" + TextUtils.join(",", ids) + ")", null);
			Log.i(TAG, "Locked " + result + " records");
		} finally {
			if (c != null) {
				c.close();
			}
			if (db != null) {
				db.close();
			}
		}
		return rs;
	}
	
	public boolean unlock(Integer id) {
		SQLiteDatabase db = new LocationOpenHelper(context).getWritableDatabase();
		db.beginTransaction();
		
		ContentValues values = new ContentValues();
		values.put("locked", 0);
		int result = db.update(LocationOpenHelper.LOCATION_TABLE_NAME, values, "id=?", new String[]{id.toString()});

		db.setTransactionSuccessful();
		db.endTransaction();
		db.close();
		
		return result == 1;
	}
	public boolean persist(JSONObject location) {
		SQLiteDatabase db = new LocationOpenHelper(context).getWritableDatabase();
		db.beginTransaction();
		ContentValues values = getContentValues(location);
		long id = db.insert(LocationOpenHelper.LOCATION_TABLE_NAME, null, values);
		
		try {
			String timestamp = location.getString("timestamp");
			Log.d(TAG, "- INSERT, id: " + id + ", timestamp: " + timestamp);
		} catch (JSONException e) {
			e.printStackTrace();
		}
		db.setTransactionSuccessful();
		db.endTransaction();
		db.close();
		
		if (id > -1) {
			return true;
		} else {
			return false;
		}
	}
	
	public void prune(Integer daysBeforeNow) {
		SQLiteDatabase db = new LocationOpenHelper(context).getWritableDatabase();
		db.beginTransaction();
		Log.i(TAG, "- PRUNE -" + daysBeforeNow + "days");
		db.delete(LocationOpenHelper.LOCATION_TABLE_NAME, "datetime(timestamp) < datetime('now', '-" + daysBeforeNow + " day')", null);
		db.setTransactionSuccessful();
		db.endTransaction();
		db.close();
	}
	public boolean destroy(Integer id) {
		SQLiteDatabase db = new LocationOpenHelper(context).getWritableDatabase();
		db.beginTransaction();
		Integer result = db.delete(LocationOpenHelper.LOCATION_TABLE_NAME, "id = ?", new String[]{id.toString()});
		if (result == 1) {
			Log.i(TAG, "- DELETE SUCCESS: " + id);
		} else {
			Log.i(TAG, "- DELETE FAILURE: " + id);
		}
		db.setTransactionSuccessful();
		db.endTransaction();
		db.close();
		return (result == 1);
	}
	
	@Override
	public void destroyAll(List<TSLocation>rs) {
		List<Integer> ids = new ArrayList<Integer>();
		for (TSLocation location : rs) {
			ids.add(location.id);
		}
		SQLiteDatabase db = new LocationOpenHelper(context).getWritableDatabase();
		db.beginTransaction();
		Integer result = db.delete(LocationOpenHelper.LOCATION_TABLE_NAME, "id IN (" + TextUtils.join(",", ids) + ")", null);
		if (result == rs.size()) {
			Log.i(TAG, "- DELETE SUCCESS (" + result + ")");
		} else {
			Log.i(TAG, "- DELETE FAILURE (" + result + ")");
		}
		db.setTransactionSuccessful();
		db.endTransaction();
		db.close();
	}

	private TSLocation hydrate(Cursor c) {
		return new TSLocation(this, c.getInt(c.getColumnIndex("id")), c.getString(c.getColumnIndex("timestamp")), c.getString(c.getColumnIndex("json")));
	}
	
	private ContentValues getContentValues(JSONObject location) {
		ContentValues values = new ContentValues();
		try {
			values.put("json", location.toString());
			values.put("timestamp", location.getString("timestamp"));
			values.put("locked", 0);
		} catch (JSONException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return values;
	}
}