package com.transistorsoft.locationmanager;

import java.sql.Date;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;
import java.util.concurrent.TimeUnit;

import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.DefaultHttpClient;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GooglePlayServicesUtil;
import com.google.android.gms.common.api.GoogleApiClient;
import com.google.android.gms.location.ActivityRecognition;
import com.google.android.gms.location.ActivityRecognitionResult;
import com.google.android.gms.location.DetectedActivity;
import com.google.android.gms.location.Geofence;
import com.google.android.gms.location.GeofencingEvent;
import com.google.android.gms.location.LocationRequest;
import com.google.android.gms.location.LocationServices;
import com.transistorsoft.locationmanager.data.LocationDAO;
import com.transistorsoft.locationmanager.data.TSLocation;
import com.transistorsoft.locationmanager.data.sqlite.SQLiteLocationDAO;

import de.greenrobot.event.EventBus;

import android.app.PendingIntent;
import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.location.Location;
import android.media.AudioManager;
import android.media.ToneGenerator;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.AsyncTask;
import android.os.Build;
import android.os.Bundle;
import android.os.IBinder;
import android.provider.Settings.Secure;
import android.util.Log;

public class BackgroundGeolocationService extends Service implements GoogleApiClient.ConnectionCallbacks, GoogleApiClient.OnConnectionFailedListener {
    private static final String TAG = "TSLocationManager";
    public static final String DATE_FORMAT = "yyyy-MM-dd'T'HH:mm:ss'Z'";
    
    private static BackgroundGeolocationService instance = null;
    
    public static boolean isInstanceCreated() {
       return instance != null;
    }
    
    // Flags
    public static final int FORCE_RELOAD_LOCATION_CHANGE 	= 1;
    public static final int FORCE_RELOAD_MOTION_CHANGE 		= 2;
    public static final int FORCE_RELOAD_GEOLOCATION 		= 3;
    
    // Actions
    public static final String ACTION_SET_CONFIG 		= "setConfig";
    public static final String ACTION_GET_LOCATIONS 	= "getLocations";
    public static final String ACTION_CHANGE_PACE 		= "changePace";
    public static final String ACTION_ON_MOTION_CHANGE    = "onMotionChange";
    public static final String ACTION_ON_GEOFENCE       = "onGeofence";
    public static final String ACTION_SYNC 				= "sync";
    public static final String ACTION_GET_ODOMETER  	= "getOdometer";
    public static final String ACTION_RESET_ODOMETER  	= "resetOdometer";
    public static final String ACTION_ADD_GEOFENCE  	= "addGeofence";
    public static final String ACTION_GET_GEOFENCES  	= "getGeofences";
    public static final String ACTION_REMOVE_GEOFENCE 	= "removeGeofence";
    public static final String ACTION_PLAY_SOUND 		= "playSound";
    public static final String ACTION_GET_CURRENT_POSITION = "getCurrentPosition";
    public static final String ACTION_GOOGLE_PLAY_SERVICES_CONNECT_ERROR = "googlePlayServiceConnectError";
    
    private LocationDAO locationDAO;
    private GoogleApiClient googleApiClient;
    private ToneGenerator toneGenerator;
    
    private PendingIntent activityRecognitionPI;
    private PendingIntent locationUpdatePI;
    private PendingIntent singleLocationUpdatePI;
    
    private Map<String, TSGeofence> geofences = new HashMap<String, TSGeofence>();
    
    private LocationRequest locationRequest;
    private LocationRequest singleLocationRequest;
    
    // Common config
    /**
     * @config {Integer} desiredAccuracy
     */
    private Integer desiredAccuracy                 = 10;
    /**
     * @config {Float} distanceFilter
     */
    private Float distanceFilter                    = 50f;
    /**
     * @config {Boolean} isDebugging
     */
    private Boolean isDebugging                     = false;
    /**
     * @config {Boolean} stopOnTerminate
     */
    private Boolean stopOnTerminate                 = false;
    
    // Android-only config
    /**
     * @config {Integer} locationUpdateInterval (ms)
     */
    private Integer locationUpdateInterval          = 60000;
    /**
     * @config {Integer} fastestLocationUpdateInterval (ms)
     */
    private Integer fastestLocationUpdateInterval   = 30000;
    /**
     * @config {Integer{ activityRecognitionInterval (ms)
     */
    private long activityRecognitionInterval     	= 60000;
    /**
     * @config {Integer} minimumActivityRecognitionConfidence
     */
    private Integer minimumActivityRecognitionConfidence = 80;
    /**
     * @config {Integer} triggerActivities
     */
    private ArrayList<Integer> triggerActivities;// = new ArrayList<Integer>();
    /*
     * @config {Boolean} forceReloadOnLocationChange Whether to reboot the Android Activity when location changes
     */
    private Boolean forceReloadOnLocationChange   	= false;
    /*
     * @config {Boolean} forceReloadOnMotionChange Whether to reboot the Android Activity when detected to have closed
     */
    private Boolean forceReloadOnMotionChange       = false;
    /*
     * @config {Boolean} forceReloadOnGeofence Whether to reboot the Android Activity when a geofence is crossed
     */
    private Boolean forceReloadOnGeofence       	= false;
    /**
     * @config {Integer} stopAfterElapsedMinutes
     */
    private Integer stopAfterElapsedMinutes 		= 0;
    /**
     * @config {Integer} stopTimeout The time to wait after ARS STILL to turn of GPS
     */
    private long stopTimeout                       = 0;
     
    // HTTP config
    /**
     * @config {Integer} maxDaysToPersist
     */
    private Integer maxDaysToPersist                = 1;
    
    /**
     * @config {String} url For sending location to your server
     */
    private String url                              = null;
    /**
     * @config {JSONObject} params For sending location to your server
     */
    private JSONObject params                       = new JSONObject();
    /**
     * @config {JSONObject} headers For sending location to your server
     */
    private JSONObject headers                      = new JSONObject();
    /**
     * @config {Boolean} autoSync
     */
    private Boolean autoSync 						= true;
    /**
     * @config {Boolean} batchSyc
     */
    private Boolean batchSync 						= false;
    
    /**
     * @property {Double} odometer
     */
    private float odometer 							= 0;
    
    // Flags
    private Boolean isEnabled           = false;
    private Boolean isMoving            = null;
    private Boolean motionStateChanged  = false;
    private Boolean isLoadingActivity   = false;
    private Boolean isUpdatingLocation  = false;
    private Boolean isAcquiringCurrentPosition = false;
    
    private long stoppedAt              = 0;
    
    private Calendar lastPrunedAt;
    private Date stopUpdatingLocationAt;
    private Location stationaryLocation;
    private Location lastLocation;
    private DetectedActivity currentActivity;
    private Bundle syncResponse;
    
    private Intent startCommandIntent;
    
    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
    	startCommandIntent = intent;
    	
        instance = this;
        
        EventBus eventBus = EventBus.getDefault();
        if (!eventBus.isRegistered(this)) {
            eventBus.register(this);
        }
        
        isEnabled = true;
        
        Log.i(TAG, "----------------------------------------");
        Log.i(TAG, "- Start BackgroundGeolocationService");
        applyConfig();
        Log.i(TAG, "----------------------------------------");
        
        // For debug sounds, turn on ToneGenerator.
        if (isDebugging) {
            toneGenerator = new ToneGenerator(AudioManager.STREAM_NOTIFICATION, 100);
        }

        // Connect to google-play services.
        if (ConnectionResult.SUCCESS == GooglePlayServicesUtil.isGooglePlayServicesAvailable(this)) {
            Log.i(TAG, "- Connecting to GooglePlayServices...");

            googleApiClient = new GoogleApiClient.Builder(this)
                .addApi(LocationServices.API)
                .addApi(ActivityRecognition.API)
                .addConnectionCallbacks(this)
                .addOnConnectionFailedListener(this)
                .build();
            
            googleApiClient.connect();
        } else {
            int resultCode = GooglePlayServicesUtil.isGooglePlayServicesAvailable(getApplicationContext());
            Log.e(TAG,  "- GooglePlayServices unavailable.  resultCode: " + resultCode);
            showErrorDialog(resultCode);
        }
        
        // Create DAO instance for persistence;
        locationDAO = new SQLiteLocationDAO(this.getApplicationContext());
        
        // Prune our database of stale records whenever service starts.
        locationDAO.prune(maxDaysToPersist);
        lastPrunedAt = Calendar.getInstance();
        
        return (stopOnTerminate) ? Service.START_NOT_STICKY : Service.START_STICKY;
    }
    
    void showErrorDialog(int code) {
    	Bundle event = new Bundle();
    	event.putString("name", ACTION_GOOGLE_PLAY_SERVICES_CONNECT_ERROR);
    	event.putInt("errorCode", code);
    	EventBus.getDefault().post(event);
	}
    
    private void applyConfig() {
    	// Load config settings
        SharedPreferences settings = getSharedPreferences(TAG, 0);
       
        if (settings.contains("debug")) {
        	isDebugging = settings.getBoolean("debug", isDebugging);
        }
        if (settings.contains("distanceFilter")) {
        	distanceFilter = settings.getFloat("distanceFilter", distanceFilter);
        }
        if (settings.contains("desiredAccuracy")) {
        	desiredAccuracy = settings.getInt("desiredAccuracy", desiredAccuracy);
        }
        if (settings.contains("locationUpdateInterval")) {
        	locationUpdateInterval = settings.getInt("locationUpdateInterval", locationUpdateInterval);
        }
        if (settings.contains("minimumActivityRecognitionConfidence")) {
        	minimumActivityRecognitionConfidence = settings.getInt("minimumActivityRecognitionConfidence", minimumActivityRecognitionConfidence);
        }
        if (settings.contains("triggerActivities")) {
        	triggerActivities = translateTriggerActivities(settings.getString("triggerActivities", ""));
        } else {
        	triggerActivities = translateTriggerActivities("in_vehicle, on_bicycle, on_foot, running, walking");
        }
        if (settings.contains("fastestLocationUpdateInterval")) {
        	fastestLocationUpdateInterval = settings.getInt("fastestLocationUpdateInterval", fastestLocationUpdateInterval);
        }        
        if (settings.contains("activityRecognitionInterval")) {
        	activityRecognitionInterval = settings.getLong("activityRecognitionInterval", activityRecognitionInterval);
        }
        if (settings.contains("stopAfterElapsedMinutes")) {
        	stopAfterElapsedMinutes = settings.getInt("stopAfterElapsedMinutes", stopAfterElapsedMinutes);
        	if (stopAfterElapsedMinutes > 0) {
            	stopUpdatingLocationAt = new Date(System.currentTimeMillis() + (stopAfterElapsedMinutes*60*1000));
            }
        }        
        if (settings.contains("stopTimeout")) {
        	stopTimeout = settings.getLong("stopTimeout", stopTimeout);
        }        
        if (settings.contains("stopOnTerminate")) {
        	stopOnTerminate = settings.getBoolean("stopOnTerminate", stopOnTerminate);
        }
        if (settings.contains("forceReloadOnLocationChange")) {
        	forceReloadOnLocationChange = settings.getBoolean("forceReloadOnLocationChange", forceReloadOnLocationChange);
        }
        if (settings.contains("forceReloadOnMotionChange")) {
        	forceReloadOnMotionChange = settings.getBoolean("forceReloadOnMotionChange", forceReloadOnMotionChange);
        }
        if (settings.contains("forceReloadOnGeofence")) {
        	forceReloadOnGeofence = settings.getBoolean("forceReloadOnGeofence", forceReloadOnGeofence);
        }
        if (settings.contains("distanceFilter")) {
        	distanceFilter = settings.getFloat("distanceFilter", distanceFilter);
        }
        
        // HTTP Configuration
        if (settings.contains("url")) {
        	url = settings.getString("url", null);
        } 
        if (settings.contains("batchSync")) {
        	batchSync = settings.getBoolean("batchSync", batchSync);
        }
        if (settings.contains("autoSync")) {
        	autoSync = settings.getBoolean("autoSync", autoSync);
        }
        if (settings.contains("params")) {
            try {
                params = new JSONObject(settings.getString("params", "{}"));
            } catch (JSONException e) {
                Log.w(TAG, "- Faile to parse #params to JSONObject");
            }
        }
        if (settings.contains("headers")) {
            try {
                headers = new JSONObject(settings.getString("headers", "{}"));
            } catch (JSONException e) {
                Log.w(TAG, "- Failed to parse #headers to JSONObject");
            }
        }
        Log.i(TAG, "  debug: " + isDebugging);
        Log.i(TAG, "  distanceFilter: " + distanceFilter);
        Log.i(TAG, "  desiredAccuracy: " + desiredAccuracy);
        Log.i(TAG, "  locationUpdateInterval: " + locationUpdateInterval);
        Log.i(TAG, "  fastestLocationUpdateInterval: " + fastestLocationUpdateInterval);
        Log.i(TAG, "  activityRecognitionInterval: " + activityRecognitionInterval);
        Log.i(TAG, "  triggerActivities: " + triggerActivities);
        Log.i(TAG, "  minimumActivityRecognitionConfidence: " + minimumActivityRecognitionConfidence);
        Log.i(TAG, "  stopTimeout: " + stopTimeout);
        Log.i(TAG, "  stopOnTerminate: " + stopOnTerminate);
        Log.i(TAG, "  forceReloadOnLocationChange: " + forceReloadOnLocationChange);
        Log.i(TAG, "  forceReloadOnMotionChange: " + forceReloadOnMotionChange);
        Log.i(TAG, "  forceReloadOnGeofence: " + forceReloadOnGeofence);
        Log.i(TAG, "  isMoving: " + isMoving);
        Log.i(TAG, "  url: " + url);
        Log.i(TAG, "  batchSync: " + batchSync);
        Log.i(TAG, "  autoSync: " + autoSync);
        if (stopAfterElapsedMinutes > 0) {
        	Log.i(TAG, "  stopAfterElapsedMinutes: " + stopAfterElapsedMinutes);
        }
    }

    @Override
    public IBinder onBind(Intent intent) {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public void onConnectionFailed(ConnectionResult arg0) {
        // TODO Auto-generated method stub
    }

    @Override
    public void onConnected(Bundle arg0) {
        Log.i(TAG, "- GooglePlayServices connected");
        
        Intent arsIntent = new Intent(this, ActivityRecognitionService.class);
        activityRecognitionPI = PendingIntent.getService(this, 0, arsIntent, PendingIntent.FLAG_CANCEL_CURRENT);
        
        Intent locationIntent = new Intent(this, LocationService.class);
        locationUpdatePI = PendingIntent.getService(this, 0, locationIntent, PendingIntent.FLAG_CANCEL_CURRENT);
        singleLocationUpdatePI = PendingIntent.getService(this, 1, locationIntent, PendingIntent.FLAG_CANCEL_CURRENT);
        
        SharedPreferences settings = getSharedPreferences(TAG, 0);
        
        // TODO Clean this up.  This is if service is started while plugin is stopped and we Activiy requests a single updateCurrentPosition.
        if (startCommandIntent != null && startCommandIntent.hasExtra("command")) {
        	String command = startCommandIntent.getStringExtra("command");
        	if (command.equalsIgnoreCase(ACTION_GET_CURRENT_POSITION)) {
        		this.onGetCurrentPosition();
        		return;
        	}
        }
        if (!stopOnTerminate && settings.contains("isMoving")) {
        	isMoving = settings.getBoolean("isMoving", false);
        	setPace(isMoving);
        } else { 
        	setPace(false);
        }

        // Start monitoring ARS
        if (googleApiClient.isConnected()) {
            requestActivityUpdates();
            hydrateGeofences();
            // Request a single activity-update if Activity is running.
            if (isMainActivityActive()) {
        		this.onGetCurrentPosition();
        	}
        }
    }

    private void onSync() {
    	if (syncResponse != null) {
    		Bundle response = new Bundle();
    		response.putString("name", ACTION_SYNC);
        	response.putBoolean("response", true);
        	response.putBoolean("success", false);
        	response.putString("message", "A sync action is already pending");
        	EventBus.getDefault().post(response);
        	return;
    	}
    	Bundle response = new Bundle();
    	response.putString("name", ACTION_SYNC);
    	response.putBoolean("response", true);
    	response.putBoolean("success", false);
    	response.putString("message", "");
    	
    	if (url != null) {
			if (isNetworkAvailable()) {
				List<TSLocation> rs = locationDAO.allWithLocking();
    			JSONArray data = new JSONArray();
    			response.putString("data", data.toString());
    			if (!rs.isEmpty()) {
    				syncResponse = response;
    				schedulePostLocations(rs);
    				for (TSLocation location : rs) {
                        data.put(location.json);
                    }
    			} else {
    				// No locations to sync, just fire back a true response with empty-array
    				response.putBoolean("success", true);
    				EventBus.getDefault().post(response);
    			}
			} else {
				response.putString("message", "No network connection");
				EventBus.getDefault().post(response);
			}
        } else {
        	List<TSLocation> rs = locationDAO.all();
        	locationDAO.destroyAll(rs);
        	JSONArray data = new JSONArray();
			if (!rs.isEmpty()) { 
				for (TSLocation location : rs) {
                    data.put(location.json);
				}
			}
			response.putString("data", data.toString());
        	response.putBoolean("success", true);
        	EventBus.getDefault().post(response);
        }
    	
    }
    private void onSetConfig(Bundle event) {
    	removeActivityUpdates();
        removeLocationUpdates();
        
        Log.i(TAG, "----------------------------------------");
        Log.i(TAG, "- reconfigure BackgroundGeolocationService");
		this.applyConfig();
		Log.i(TAG, "----------------------------------------");
		
		requestActivityUpdates();
		if (isMoving) {
			requestLocationUpdates();
		}
    }
    private void onAddGeofence(Bundle event) {
    	Log.i(TAG, "- onAddGeofence: " + event);
    	
    	String identifier = event.getString("identifier");
    	if (geofences.containsKey(identifier)) {
    		removeGeofence(identifier);
    	}
    	boolean notifyOnEnter = false;
	    boolean notifyOnExit = false;
	    if (event.containsKey("notifyOnEnter")) {
	    	notifyOnEnter = event.getBoolean("notifyOnEnter");
	    }
	    if (event.containsKey("notifyOnExit")) {
	    	notifyOnExit = event.getBoolean("notifyOnExit");
	    }
    	TSGeofence geofence = new TSGeofence(identifier, event.getFloat("radius"), event.getDouble("latitude"), event.getDouble("longitude"), notifyOnEnter, notifyOnExit);
    	geofences.put(geofence.identifier, geofence);
    	persistGeofences();
        LocationServices.GeofencingApi.addGeofences(googleApiClient, geofence.getGeofencingRequest(), geofence.getGeofencePendingIntent(this));
    }
    private void onGetGeofences(Bundle event) {
    	Log.i(TAG, "- onGetGeofences: " + event);
    	
    	Bundle response = new Bundle();
    	response.putBoolean("response", true);
    	response.putString("name", event.getString("name"));
    	
    	JSONArray json = new JSONArray();
    	SharedPreferences settings = getSharedPreferences(TAG, 0);
    	if (settings.contains("geofences")) {
    		try {
    			json = new JSONArray(settings.getString("geofences", "[]"));
			} catch (JSONException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
    	}
    	response.putString("data", json.toString());
    	EventBus.getDefault().post(response);
    }
    private void onSetPace(Bundle event) {
    	Boolean moving = event.getBoolean("isMoving"); 
    	setPace(moving);
    	Bundle response = new Bundle();
    	response.putString("name", ACTION_CHANGE_PACE);
    	response.putBoolean("response", true);
    	response.putBoolean("isMoving", moving);
    	EventBus.getDefault().post(response);
    }
    private void onGetLocations() {
    	Bundle response = new Bundle();
    	response.putString("name", ACTION_GET_LOCATIONS);
    	response.putBoolean("response", true);
    	List<TSLocation> rs = locationDAO.all();
    	
    	JSONArray data = new JSONArray();
    	for (TSLocation location : rs) {
            data.put(location.json);
        }
    	response.putString("data", data.toString());
    	EventBus.getDefault().post(response);
    }
    private void onGetOdometer() {
    	Bundle response = new Bundle();
    	response.putString("name", ACTION_GET_ODOMETER);
    	response.putBoolean("response", true);
    	response.putBoolean("success", true);
    	response.putFloat("data", odometer);
    	EventBus.getDefault().post(response);
    }
    
    private void onGetCurrentPosition() {
    	isAcquiringCurrentPosition = true;
    	requestSingleLocationUpdate();
    }
    
    private void onResetOdometer() {
    	odometer = 0;
    	Bundle response = new Bundle();
    	response.putString("name", ACTION_RESET_ODOMETER);
    	response.putBoolean("response", true);
    	response.putBoolean("success", true);
    	EventBus.getDefault().post(response);
    }
    
    private void playSound(int soundId) {
        int duration = 1000;
        if (toneGenerator == null) {
        	toneGenerator = new ToneGenerator(AudioManager.STREAM_NOTIFICATION, 100);
        }
        toneGenerator.startTone(soundId, duration);
    }
    
    /**
     * Activity -> Service event-handler
     * @param {Bundle} event
     */
    public void onEventMainThread(Bundle event) {    	
    	if (event.containsKey("response")) {
    		return;
    	}
    	Log.i(TAG, "- command received: " + event);
    	String name = event.getString("name");
    	
    	if (ACTION_SET_CONFIG.equalsIgnoreCase(name)) {
    		this.onSetConfig(event);
    	}
    	else if (ACTION_GET_LOCATIONS.equalsIgnoreCase(name)) {
    		this.onGetLocations();
    	} else if (ACTION_CHANGE_PACE.equalsIgnoreCase(name)) {
    		this.onSetPace(event);
    	} else if (ACTION_SYNC.equalsIgnoreCase(name)) {
    		this.onSync();
    	} else if (ACTION_GET_ODOMETER.equalsIgnoreCase(name)) {
    		this.onGetOdometer();
    	} else if (ACTION_RESET_ODOMETER.equalsIgnoreCase(name)) {
    		this.onResetOdometer();
    	} else if (ACTION_ADD_GEOFENCE.equalsIgnoreCase(name)) {
    		this.onAddGeofence(event);
    	} else if (ACTION_REMOVE_GEOFENCE.equalsIgnoreCase(name)) {
    		this.removeGeofence(event.getString("identifier"));
    	} else if (ACTION_GET_GEOFENCES.equalsIgnoreCase(name)) {
    		this.onGetGeofences(event);
    	} else if (ACTION_PLAY_SOUND.equalsIgnoreCase(name)) {   
    		this.playSound(event.getInt("soundId"));
    	} else if (ACTION_GET_CURRENT_POSITION.equalsIgnoreCase(name)) {
    		this.onGetCurrentPosition();
    	}
    }

    /**
     * EventBus listener for ARS
     * @param {ActivityRecognitionResult} result
     */
    public void onEventMainThread(ActivityRecognitionResult result) {
        currentActivity 	= result.getMostProbableActivity();
        String activityName = getActivityName(currentActivity.getType());
        int confidence 		= currentActivity.getConfidence();
        
        Log.i(TAG, "- Activity received: " + activityName + ", confidence: " + confidence);
        
        if (stopUpdatingLocationAt != null) {
        	Date now = new Date(result.getTime());
        	if (now.after(stopUpdatingLocationAt)) {
        		Log.i(TAG, "- Elapsed minutes has expired.  Stopping service");
        		stopUpdatingLocationAt = null;
        		stopSelf();
        		return;
        	}
        }
        
        // If configured to stop when user closes app, kill this service.
        if (!isMainActivityActive() && stopOnTerminate) {
            stopSelf();
            return;
        }
        
        boolean wasMoving 		= isMoving;
        boolean nowMoving 		= triggerActivities.contains(currentActivity.getType());
        boolean startedMoving   = !wasMoving && nowMoving;
        boolean justStopped     = wasMoving && !nowMoving;
        boolean initialState    = !nowMoving && (stationaryLocation == null);
        
        // If we're using a stopTimeout, record the current activity's timestamp.
        if (justStopped && stopTimeout > 0 && stoppedAt == 0) {
            stoppedAt = result.getElapsedRealtimeMillis();
            return;
        }
        
        // If we're using a stopTimeout, compare the current activity's timestamp with the 1st recorded STILL event.
        if (!nowMoving && stoppedAt > 0) {
            long elapsedMillis = result.getElapsedRealtimeMillis() - stoppedAt;
            long elapsedMinutes = TimeUnit.MILLISECONDS.toMinutes(elapsedMillis);
            Log.i(TAG, "- Waiting for stopTimeout (" + stopTimeout + " min): elapsed min: " + elapsedMinutes);
            if (elapsedMinutes >= stopTimeout) {
                justStopped = true;
            } else {
                return;
            }
        }
        stoppedAt = 0;
        
        // State-changes require the confidence to meet or exceed the configured minimum-confidence.
        if ( (startedMoving || justStopped || initialState) && (confidence >= minimumActivityRecognitionConfidence)) {
        	setPace(nowMoving);
        }
        // Check our location-queue and sync if not empty.
    	if (url != null && autoSync && isNetworkAvailable()) {
    		List<TSLocation> rs = locationDAO.allWithLocking();
			if (!rs.isEmpty()) {
				schedulePostLocations(rs);
			}
    	}
    }
    
    public void onEventMainThread(Location location) {
    	Log.i(TAG, "BUS Rx:" + location.toString());
        startTone("beep");
        
    	// Update Odometer
        if (lastLocation != null) {
        	odometer = odometer + location.distanceTo(lastLocation);
        }
    	lastLocation = location;
    	
    	if (isEnabled) {
    		persistLocation(location);
    	}
        
        // Check if we're due for pruning
    	Calendar now = Calendar.getInstance();
    	long daysSincePrune = (now.getTimeInMillis() - lastPrunedAt.getTimeInMillis()) / (24 * 60 * 60 * 1000);
    	if (daysSincePrune >= 1) {
    		locationDAO.prune(maxDaysToPersist);
    		lastPrunedAt = now;
    	}
    	
    	// Stop updating single-location PendingIntent if we're fetching teh current position.
    	if (isAcquiringCurrentPosition) {
    		Log.i(TAG, "- current-position received");
    		isAcquiringCurrentPosition = false;
    		LocationServices.FusedLocationApi.removeLocationUpdates(googleApiClient, singleLocationUpdatePI);    		
    		return;
    	}
    	// Check if we're force-loading the Activity at-the-moment -- we don't want to execute this twice simultaneously.
    	Boolean isActivityActive = isMainActivityActive(); 
    	if (isLoadingActivity) {
    		return;
    	}
    	
    	if (motionStateChanged) {
    		fireMotionStateChanged(location);
    	} else if (forceReloadOnLocationChange && !isActivityActive) {
    		Bundle response = new Bundle();
        	response.putString("location", locationToJson(location, currentActivity, isMoving).toString());
        	forceMainActivityReload(FORCE_RELOAD_LOCATION_CHANGE, response);
    	}
    }

    public void onEventMainThread(GeofencingEvent geofenceEvent) {
    	Log.i(TAG, "- Rx GeofencingEvent: " + geofenceEvent);
    	
    	// Check if we're force-loading the Activity at-the-moment -- we don't want to execute this twice simultaneously. 
    	if (forceReloadOnGeofence && !isMainActivityActive() && !isLoadingActivity) {
    		Bundle event = new Bundle();
    		event.putString("geofencingEvent", geofencingEventToJson(geofenceEvent, geofenceEvent.getTriggeringGeofences().get(0)).toString());
    		forceMainActivityReload(FORCE_RELOAD_GEOLOCATION, event);
    	}
    	startTone("chirp_chirp_chirp");
    }
    
    private void setPace(Boolean moving) {
    	if (!isEnabled) {
    		return;
    	}
        Log.i(TAG, "- setPace: " + moving);
        Boolean wasMoving   	= isMoving;
        isMoving            	= moving;
        motionStateChanged      = false;
        
        // Cache our isMoving state in preferences in case our service is restarted.
        SharedPreferences settings = getSharedPreferences("TSLocationManager", 0);
        SharedPreferences.Editor editor = settings.edit();
        editor.putBoolean("isMoving", isMoving);
        editor.commit();

    	if (wasMoving != isMoving) {
    		motionStateChanged = true;
    	}
    	
    	if (moving) {
    		startTone("doodly_doo");
            stationaryLocation = null;
            requestLocationUpdates();
        } else {
            removeLocationUpdates();
            startTone("long_beep");
            		
            // set our stationaryLocation
            stationaryLocation = LocationServices.FusedLocationApi.getLastLocation(googleApiClient);
            Log.i(TAG, "- stationaryLocation: " + stationaryLocation);
            Log.i(TAG, "- motionStateChagned: " + motionStateChanged);
            if (stationaryLocation != null) {
            	if (motionStateChanged) {
            		persistLocation(stationaryLocation);
            		fireMotionStateChanged(stationaryLocation);
            	}
            } else {
            	Log.w(TAG,  "- Failed to obtain stationary-location");
            }
        }
    }
    private void fireMotionStateChanged(Location location) {
    	Bundle response = new Bundle();
    	response.putString("name", ACTION_ON_MOTION_CHANGE);
    	response.putBoolean("response", true);
    	response.putBoolean("success", true);
    	response.putBoolean("isMoving", isMoving);
    	response.putString("location", locationToJson(location, currentActivity, isMoving).toString());
    	
		// Force main-activity reload (if not running) if we're detected to be moving.
        if (!isMainActivityActive()) {
        	if (forceReloadOnMotionChange) {
        		forceMainActivityReload(FORCE_RELOAD_MOTION_CHANGE, response);
        	}
        } else {
        	EventBus.getDefault().post(response);
        }
        motionStateChanged = false;
    }
    /**
     * Translates a number representing desired accuracy of GeoLocation system from set [0, 10, 100, 1000].
     * 0:  most aggressive, most accurate, worst battery drain
     * 1000:  least aggressive, least accurate, best for battery.
     */
    private Integer translateDesiredAccuracy(Integer accuracy) {
        switch (accuracy) {
            case 1000:
                accuracy = LocationRequest.PRIORITY_NO_POWER;
                break;
            case 100:
                accuracy = LocationRequest.PRIORITY_LOW_POWER;
                break;
            case 10:
                accuracy = LocationRequest.PRIORITY_BALANCED_POWER_ACCURACY;
                break;
            case 0:
                accuracy = LocationRequest.PRIORITY_HIGH_ACCURACY;
                break;
            default:
                accuracy = LocationRequest.PRIORITY_BALANCED_POWER_ACCURACY;
        }
        return accuracy;
    }
    
    private Integer getLocationUpdateInterval() {
        // TODO Can add intelligence here based upon currentActivity.
        SharedPreferences settings = getSharedPreferences(TAG, 0);
        return settings.getInt("locationUpdateInterval", locationUpdateInterval);
    }
    
    private Integer getFastestLocationUpdateInterval() {
        /* TODO Add intelligent calculation of fastestLocationUpdateInterval based upon currentActivity here
         * switch (currentActivity.getType()) {
            case DetectedActivity.IN_VEHICLE:
                fastestLocationUpdateInterval = 30000;
                break;
            case DetectedActivity.ON_BICYCLE:
                fastestLocationUpdateInterval = 30000;
                break;
            case DetectedActivity.ON_FOOT:
                fastestLocationUpdateInterval = 30000;
                break;
            case DetectedActivity.RUNNING:
                fastestLocationUpdateInterval = 30000;
                break;
            case DetectedActivity.WALKING:
                fastestLocationUpdateInterval = 30000; 
                break;
        }
        */
        return fastestLocationUpdateInterval;
    }
    
    /**
     * Iterate human-provided list-of-trigger-activities, eg: "on_foot, in_vehicle" -> [DetectedActivity.ON_FOOT, DetectedActivity.IN_VEHICLE]
     * @param {String} comma-delimited string of activities "on_foot, in_vehicle"
     * @return {ArrayList<Integer>}
     * 
     */
    private ArrayList<Integer> translateTriggerActivities(String activities) {
    	ArrayList<Integer> translatedActivities = new ArrayList<Integer>();
    	String foo = "a";
    	Log.i(TAG, "foo: " + foo);
    	for (String activityName : Arrays.asList(activities.replaceAll("\\s+","").split(","))) {
    		if (activityName.equalsIgnoreCase("in_vehicle")) {
	            translatedActivities.add(DetectedActivity.IN_VEHICLE);
    		} else if (activityName.equalsIgnoreCase("on_bicycle")) {
	            translatedActivities.add(DetectedActivity.ON_BICYCLE);
    		} else if (activityName.equalsIgnoreCase("on_foot")) {
	            translatedActivities.add(DetectedActivity.ON_FOOT);
    		} else if (activityName.equalsIgnoreCase("running")) {
	            translatedActivities.add(DetectedActivity.RUNNING);
    		} else if (activityName.equalsIgnoreCase("walking")) {
	            translatedActivities.add(DetectedActivity.WALKING);
    		} else {
	            Log.w(TAG, "- CONFIGURE ERROR: Unknown activity-name in #triggerActivities: " + activityName);
    		}
    	}
    	return translatedActivities;
    }
    
    public static String getActivityName(int activityType) {
        switch (activityType) {
            case DetectedActivity.IN_VEHICLE:
                return "in_vehicle";
            case DetectedActivity.ON_BICYCLE:
                return "on_bicycle";
            case DetectedActivity.ON_FOOT:
                return "on_foot";
            case DetectedActivity.RUNNING:
                return "running";
            case DetectedActivity.WALKING:
                return "walking";
            case DetectedActivity.STILL:
                return "still";
            case DetectedActivity.UNKNOWN:
                return "unknown";
            case DetectedActivity.TILTING:
                return "tilting";
        }
        return "unknown";
    }
    
    /**
     * For when booting, check persisted geofences and re-initialize them.
     */
    private void hydrateGeofences() {
    	SharedPreferences settings = getSharedPreferences(TAG, 0);
    	if (settings.contains("geofences")) {
    		try {
				JSONArray json = new JSONArray(settings.getString("geofences", "[]"));				
				for (int i = 0; i < json.length(); i++) {
				    JSONObject row = json.getJSONObject(i);
				    
				    boolean notifyOnEnter = false;
				    boolean notifyOnExit = false;
				    if (row.has("notifyOnEnter")) {
				    	notifyOnEnter = row.getBoolean("notifyOnEnter");
				    }
				    if (row.has("notifyOnExit")) {
				    	notifyOnExit = row.getBoolean("notifyOnExit");
				    }
				    TSGeofence geofence = new TSGeofence(row.getString("identifier"), (float) row.getDouble("radius"), row.getDouble("latitude"), row.getDouble("longitude"), notifyOnEnter, notifyOnExit);
				    geofences.put(geofence.identifier, geofence);
				    Log.i(TAG, "- hydrate geofence: " + geofence.identifier + ", " + notifyOnEnter + ", " + notifyOnExit + ", " + geofence);
			        LocationServices.GeofencingApi.addGeofences(googleApiClient, geofence.getGeofencingRequest(), geofence.getGeofencePendingIntent(this));
				}
			} catch (JSONException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
    	}
    }
    private void persistGeofences() {
    	SharedPreferences settings = getSharedPreferences(TAG, 0);
        SharedPreferences.Editor editor = settings.edit();
        
        JSONArray json = new JSONArray();
        
        for (Map.Entry<String, TSGeofence> entry : geofences.entrySet()) {
        	json.put(entry.getValue().asJson());
        }
        editor.putString("geofences", json.toString());
        editor.commit();
        Log.i(TAG, "- persistGeofences: " + settings.getString("geofences", "<EMPTY>"));
    }
    private void removeGeofence(String identifier) {
    	if (geofences.containsKey(identifier)) {
    		TSGeofence geofence = geofences.get(identifier);
    		LocationServices.GeofencingApi.removeGeofences(googleApiClient, geofence.pendingIntent);
    		geofences.remove(identifier);
    		persistGeofences();
    	}
    }

    private boolean isNetworkAvailable() {
        ConnectivityManager connectivityManager = (ConnectivityManager) getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo activeNetworkInfo = connectivityManager.getActiveNetworkInfo();
        return activeNetworkInfo != null && activeNetworkInfo.isConnected();
    }
    
    private boolean persistLocation(Location location) {
        if (locationDAO.persist(locationToJson(location, currentActivity, isMoving))) {
        	if (url != null && autoSync && isNetworkAvailable()) {
            	schedulePostLocations(locationDAO.allWithLocking());
            }
            return true;
        } else {
            Log.w(TAG, "INSERT FAILURE" + location);
            return false;
        }
    }
    
    private void schedulePostLocations(List<TSLocation>rs) {
    	
        PostLocationTask task = new BackgroundGeolocationService.PostLocationTask();
        task.setBatchMode(batchSync);
        task.setRecords(rs);
        
        Log.d(TAG, "- beforeexecute " +  task.getStatus());

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.HONEYCOMB)
            task.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR);
        else
            task.execute();
        Log.d(TAG, "- afterexecute " +  task.getStatus());
    }
    
    private boolean doBatchPost(JSONArray data) {
    	try {
    		params.put("location", data);
    		DefaultHttpClient httpClient = new DefaultHttpClient();
            HttpPost request = createHttpRequest();

            Log.d(TAG, "POST batch (" + data.length() + " records)");
            HttpResponse response = httpClient.execute(request);
            Log.d(TAG, "RESPONSE received: " + response.getStatusLine());
            
            return handleResponse(response);
        } catch (Throwable e) {
            Log.w(TAG, "EXCEPTION posting location: " + e);
            e.printStackTrace();
            return false;
        }
    }
    
    private boolean doPost(JSONObject data) {
        try {
        	params.put("location", data);
        	DefaultHttpClient httpClient = new DefaultHttpClient();
            HttpPost request = createHttpRequest();
 
            Log.d(TAG, "POST " + data.getString("timestamp"));
            HttpResponse response = httpClient.execute(request);
            Log.d(TAG, "RESPONSE received: " + data.getString("timestamp") + ", " + response.getStatusLine());
            
            return handleResponse(response);
        } catch (Throwable e) {
            Log.w(TAG, "EXCEPTION posting location: " + e);
            e.printStackTrace();
            return false;
        }
    }
    
    private HttpPost createHttpRequest() {
    	try {
    		// Append android UUID to params so that server can map the UUID to some user in your database on server.
            params.put("device_id", Secure.getString(this.getContentResolver(), Secure.ANDROID_ID));
            
	    	HttpPost request = new HttpPost(url);
	        StringEntity se = new StringEntity(params.toString());
	        request.setEntity(se);
	        request.setHeader("Accept", "application/json");
	        request.setHeader("Content-type", "application/json");
	        
	        Iterator<String> keys = headers.keys();
	        while( keys.hasNext() ){
	            String key = keys.next();
	            if(key != null) {
	                request.setHeader(key, (String)headers.getString(key));
	            }
	        }
	        return request;
    	} catch (Throwable e) {
            Log.w(TAG, "EXCEPTION posting location: " + e);
            e.printStackTrace();
            return null;
        }
    }
    
    private boolean handleResponse(HttpResponse response) {
    	// Response code
        switch (response.getStatusLine().getStatusCode()) {
        	case 200:
        	case 201:
        	case 204:
        		return true;
        	case 500:
        	case 404:
        		return false;
        	default:
        		return false;
        }
    }
    
    private Boolean isMainActivityActive() {
    	SharedPreferences settings = getSharedPreferences(TAG, 0);
    	Boolean isActivityActive = settings.getBoolean("activityIsActive", false);
    	if (isLoadingActivity && isActivityActive) {
    		isLoadingActivity = false;
    	}
    	return isActivityActive;
    }
    /**
     * Forces the main activity to re-launch if it's unloaded.  This is how we're able to rely upon Javascript
     * running always, since we for the app to boot.
     * @param {int} reasonCode @see const FORCE_RELOAD_* 
     */
    private void forceMainActivityReload(int eventCode, Bundle event) {
    	isLoadingActivity = true;
        Log.w(TAG, "- Forcing main-activity reload");
        PackageManager pm = getPackageManager();
        Intent launchIntent = pm.getLaunchIntentForPackage(getApplicationContext().getPackageName());
        event.putBoolean("forceReload", true);
        event.putInt("eventCode", eventCode);
        launchIntent.putExtras(event);
        
        launchIntent.addFlags(Intent.FLAG_FROM_BACKGROUND);
        launchIntent.addFlags(Intent.FLAG_ACTIVITY_NO_USER_ACTION);
        launchIntent.addFlags(Intent.FLAG_ACTIVITY_NO_ANIMATION);
        startActivity(launchIntent);
    }
    
    private void requestActivityUpdates() {
    	Log.i(TAG, "- requesting activity updates " + activityRecognitionInterval);
        SharedPreferences settings = getSharedPreferences(TAG, 0);
        ActivityRecognition.ActivityRecognitionApi.requestActivityUpdates(googleApiClient, settings.getLong("activityRecognitionInterval", activityRecognitionInterval), activityRecognitionPI);
    }
    
    private void removeActivityUpdates() {
        ActivityRecognition.ActivityRecognitionApi.removeActivityUpdates(googleApiClient, activityRecognitionPI);
        isUpdatingLocation = false;
    }

    private void requestSingleLocationUpdate() {
    	// Configure LocationRequest
        singleLocationRequest = LocationRequest.create()
            .setPriority(LocationRequest.PRIORITY_HIGH_ACCURACY)
            .setNumUpdates(1)
            .setExpirationDuration(30000)
            .setInterval(0)
            .setFastestInterval(0)
            .setSmallestDisplacement(0);
        
        LocationServices.FusedLocationApi.requestLocationUpdates(googleApiClient, singleLocationRequest, singleLocationUpdatePI);
    }
    private void requestLocationUpdates() {
        if (!isEnabled || isUpdatingLocation) { return; }    // <-- Don't engage GPS when app is in foreground
        
        SharedPreferences settings = getSharedPreferences(TAG, 0);
        
        // Configure LocationRequest
        locationRequest = LocationRequest.create()
            .setPriority(translateDesiredAccuracy(settings.getInt("desiredAccuracy", desiredAccuracy)))
            .setInterval(getLocationUpdateInterval())
            .setFastestInterval(getFastestLocationUpdateInterval())
            .setSmallestDisplacement(settings.getFloat("distanceFilter", distanceFilter));
        
        LocationServices.FusedLocationApi.requestLocationUpdates(googleApiClient, locationRequest, locationUpdatePI);
        isUpdatingLocation = true;
    }
    
    private void removeLocationUpdates() {
        LocationServices.FusedLocationApi.removeLocationUpdates(googleApiClient, locationUpdatePI);
        isUpdatingLocation = false;
    }
    
    /**
     * Plays debug sound
     * @param name
     */
    private void startTone(String name) {
        int tone = 0;
        int duration = 1000;

        if (name.equals("beep")) {
            tone = ToneGenerator.TONE_PROP_BEEP;
        } else if (name.equals("beep_beep_beep")) {
            tone = ToneGenerator.TONE_CDMA_CONFIRM;
        } else if (name.equals("long_beep")) {
            tone = ToneGenerator.TONE_CDMA_ABBR_ALERT;
        } else if (name.equals("doodly_doo")) {
            tone = ToneGenerator.TONE_CDMA_ALERT_NETWORK_LITE;
        } else if (name.equals("chirp_chirp_chirp")) {
            tone = ToneGenerator.TONE_CDMA_ALERT_CALL_GUARD;
        } else if (name.equals("dialtone")) {
            tone = ToneGenerator.TONE_SUP_RINGTONE;
        }
        if (isDebugging) {
            toneGenerator.startTone(tone, duration);
        }
    }

    @Override
    public void onConnectionSuspended(int arg0) {
        // TODO Auto-generated method stub
    }
    
    @Override
    public void onDestroy() {
        Log.w(TAG, "- Destroy service");
        
        cleanUp();
        super.onDestroy();
    }
    private void cleanUp() {
        instance = null;
        EventBus.getDefault().unregister(this);
        
        if (googleApiClient != null && googleApiClient.isConnected()) {
            removeActivityUpdates();
            removeLocationUpdates();
            
            if (stopOnTerminate) {
            	// Remove geofences.
            	Object[] keys = geofences.keySet().toArray();
            	for (int n=0; n < keys.length;n++) {
            		removeGeofence(keys[n].toString());
            	}
            }
            googleApiClient.disconnect();
        }
    }
    
    /**
     * Convert a Location instance to JSONObject
     * @param Location
     * @return JSONObject
     */
    public static JSONObject locationToJson(Location l, DetectedActivity activity, Boolean isMoving) {
    	TimeZone tz = TimeZone.getTimeZone("UTC");
		SimpleDateFormat dateFormatter = new SimpleDateFormat(DATE_FORMAT, Locale.getDefault());
		dateFormatter.setTimeZone(tz);
		
        try {
            JSONObject data = new JSONObject();
            JSONObject coordData = new JSONObject();
            JSONObject activityData = new JSONObject();
            
            coordData.put("latitude", l.getLatitude());
            coordData.put("longitude", l.getLongitude());
            coordData.put("accuracy", l.getAccuracy());
            coordData.put("speed", l.getSpeed());
            coordData.put("heading", l.getBearing());
            coordData.put("altitude", l.getAltitude());
            
            if (activity != null) {
            	activityData.put("type", getActivityName(activity.getType()));
            	activityData.put("confidence", activity.getConfidence());
            	data.put("activity", activityData);
            }
            data.put("is_moving", isMoving);
            data.put("coords", coordData);
            data.put("timestamp", dateFormatter.format(l.getTime()));
            return data;
        } catch (JSONException e) {
            Log.e(TAG, "could not parse location");
            return null;
        }
    }
    public static JSONObject geofencingEventToJson(GeofencingEvent geofenceEvent, Geofence geofence) {
    	JSONObject params = new JSONObject();
        String action = "";
        int transitionType = geofenceEvent.getGeofenceTransition();
        if (transitionType == Geofence.GEOFENCE_TRANSITION_ENTER) {
            action = "ENTER";
        } else if (transitionType == Geofence.GEOFENCE_TRANSITION_EXIT) {
            action = "EXIT";
        } else {
            action = "DWELL";
        }
        try {
            params.put("identifier", geofence.getRequestId());
            params.put("action", action);
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return params;
    }
    private class PostLocationTask extends AsyncTask<Object, Integer, Boolean> {
        private Boolean batchMode = false;
        private List<TSLocation> records = null;
        
        public void setBatchMode(Boolean mode) {
        	batchMode = mode;
        }
        public void setRecords(List<TSLocation>_records) {
        	records = _records;
        }
        
    	@Override
        protected Boolean doInBackground(Object...objects) {
            Log.d(TAG, "Executing PostLocationTask#doInBackground");

            if (batchMode) {
            	JSONArray rs = new JSONArray();
            	for (TSLocation location : records) {
                    rs.put(location.json);
                }
            	if (doBatchPost(rs)) {            		
            		locationDAO.destroyAll(records);
            	}
            } else {
            	boolean remoteOK 	= true;
            	for (TSLocation location : records) {
        			if (remoteOK) {
            			remoteOK = doPost(location.json);
                    }
        			if (remoteOK) {
        				location.destroy();
        			} else {
                    	Log.d(TAG, "- POST FAILURE:  unlocking " + location.id);
                    	location.unlock();
                    }
                }
            }
            return true;
        }
        @Override
        protected void onPostExecute(Boolean result) {
            Log.d(TAG, "PostLocationTask#onPostExecute");
            if (syncResponse != null) {
            	syncResponse.putBoolean("success", result);
            	EventBus.getDefault().post(syncResponse);
            	syncResponse = null;
            }
        }
    }
}
